package org.eclipse.swt.graphics;

/*
 *  OS/2 version.
 *  Copyright (c) 2002, 2004 EclipseOS2 Team.
 */

/*
 *  Copyright (c) 2000, 2002 IBM Corp.  All rights reserved.
 *  This file is made available under the terms of the Common Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/cpl-v10.html
 */

import org.eclipse.swt.internal.*;
import org.eclipse.swt.internal.pm.*;
import org.eclipse.swt.*;

/**
 * Instances of this class describe operating system fonts.
 * Only the public API of this type is platform independent.
 * <p>
 * For platform-independent behaviour, use the get and set methods
 * corresponding to the following properties:
 * <dl>
 * <dt>height</dt><dd>the height of the font in points</dd>
 * <dt>name</dt><dd>the face name of the font, which may include the foundry</dd>
 * <dt>style</dt><dd>A bitwise combination of NORMAL, ITALIC and BOLD</dd>
 * </dl>
 * If extra, platform-dependent functionality is required:
 * <ul>
 * <li>On <em>Windows</em>, the <code>data</code> member of the <code>FontData</code>
 * corresponds to a Windows <code>LOGFONT</code> structure whose fields
 * may be retrieved and modified.</li>
 * <li>On <em>X</em>, the fields of the <code>FontData</code> correspond
 * to the entries in the font's XLFD name and may be retrieved and modified.</li>
 * <li>On <em>OS/2</em>, some fields of the <code>FATTRS</code> structure are
 * represented as data members of the <code>FontData</code> and may be retrieved
 * and modified.</li>
 * </ul>
 * Application code does <em>not</em> need to explicitly release the
 * resources managed by each instance when those instances are no longer
 * required, and thus no <code>dispose()</code> method is provided.
 *
 * @see Font
 */

public final class FontData {
	
	/**
	 * The family name of the font data
	 * (Warning: This field is platform dependent)
	 */
    String name;

	/**
	 * The height of the font data in points 
	 * (Warning: This field is platform dependent)
	 */
	int height;
    
	/**
	 * The style of the font data
	 * (Warning: This field is platform dependent)
	 */
    int style;

	/**
	 * The FATTRS.szFacename field of the font data
	 * (Warning: This field is platform dependent)
	 */
     public String szFacename;

	/**
	 * The FATTRS.fsSelection field of the font data
	 * (Warning: This field is platform dependent)
	 */
     public short fsSelection;
     
	/**
	 * The locales of the font
	 * (Warning: These fields are platform dependent)
	 */
	String lang, country, variant;
	
/**	 
 * Constructs a new un-initialized font data.
 */
public FontData() {
    name = "";
    szFacename = "";
}

/**
 * Constructs a new FontData given a string representation
 * in the form generated by the <code>FontData.toString</code>
 * method.
 * <p>
 * Note that the representation varies between platforms,
 * and a FontData can only be created from a string that was 
 * generated on the same platform.
 * </p>
 *
 * @param string the string representation of a <code>FontData</code> (must not be null)
 *
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_NULL_ARGUMENT - if the argument is null</li>
 *    <li>ERROR_INVALID_ARGUMENT - if the argument does not represent a valid description</li>
 * </ul>
 *
 * @see #toString
 */
public FontData(String string) {
	if (string == null) SWT.error(SWT.ERROR_NULL_ARGUMENT);
	int start = 0;
	int end = string.indexOf('|');
	if (end == -1) SWT.error(SWT.ERROR_INVALID_ARGUMENT);
	String version1 = string.substring(start, end);
	
	start = end + 1;
	end = string.indexOf('|', start);
	if (end == -1) SWT.error(SWT.ERROR_INVALID_ARGUMENT);
	String name = string.substring(start, end);
	
	start = end + 1;
	end = string.indexOf('|', start);
	if (end == -1) SWT.error(SWT.ERROR_INVALID_ARGUMENT);
	int height = 0;
	try {
		height = Integer.parseInt(string.substring(start, end));
	} catch (NumberFormatException e) {
		SWT.error(SWT.ERROR_INVALID_ARGUMENT);
	}
	
	start = end + 1;
	end = string.indexOf('|', start);
	if (end == -1) SWT.error(SWT.ERROR_INVALID_ARGUMENT);
	int style = 0;
	try {
		style = Integer.parseInt(string.substring(start, end));
	} catch (NumberFormatException e) {
		SWT.error(SWT.ERROR_INVALID_ARGUMENT);
	}

	start = end + 1;
	end = string.indexOf('|', start);
//@@TODO (dmik): remove    
//	data = new LOGFONT();
//	data.lfCharSet = OS.DEFAULT_CHARSET;
	setName(name);
	setHeight(height);
	setStyle(style);
	if (end == -1) return;
	String platform = string.substring(start, end);

	start = end + 1;
	end = string.indexOf('|', start);
	if (end == -1) return;
	String version2 = string.substring(start, end);

	if (platform.equals("OS2") && version2.equals("1")) {
        start = end + 1;
        end = string.indexOf('|', start);
        if (end == -1) return;
        szFacename = string.substring(start, end);
		try {
			start = end + 1;
            end = string.length();
            fsSelection = Short.parseShort(string.substring(start, end));
		} catch (NumberFormatException e) {
			return;
		}
	}
}

/**	 
 * Constructs a new font data given a font name,
 * the height of the desired font in points, 
 * and a font style.
 *
 * @param name the name of the font (must not be null)
 * @param height the font height in points
 * @param style a bit or combination of NORMAL, BOLD, ITALIC
 *
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_NULL_ARGUMENT - when the font name is null</li>
 *    <li>ERROR_INVALID_ARGUMENT - if the height is negative</li>
 * </ul>
 */
public FontData(String name, int height, int style) {
	if (name == null) SWT.error(SWT.ERROR_NULL_ARGUMENT);
	setName(name);
	setHeight(height);
	setStyle(style);
    szFacename = "";
}

/**
 * Compares the argument to the receiver, and returns true
 * if they represent the <em>same</em> object using a class
 * specific comparison.
 *
 * @param object the object to compare with this object
 * @return <code>true</code> if the object is the same as this object and <code>false</code> otherwise
 *
 * @see #hashCode
 */
public boolean equals (Object object) {
	if (object == this) return true;
	if (!(object instanceof FontData)) return false;
    FontData fd = (FontData) object;
	return
        name.equals (fd.name) &&
        height == fd.height &&
        style == fd.style &&
        szFacename.equals (fd.szFacename) &&
        fsSelection == fd.fsSelection;
}

/**
 * Returns the height of the receiver in points.
 *
 * @return the height of this FontData
 *
 * @see #setHeight
 */
public int getHeight() {
	return height;
}

/**
 * Returns the name of the receiver.
 * On platforms that support font foundries, the return value will
 * be the foundry followed by a dash ("-") followed by the face name.
 *
 * @return the name of this <code>FontData</code>
 *
 * @see #setName
 */
public String getName() {
	return name;
}

/**
 * Returns the style of the receiver which is a bitwise OR of 
 * one or more of the <code>SWT</code> constants NORMAL, BOLD
 * and ITALIC.
 *
 * @return the style of this <code>FontData</code>
 * 
 * @see #setStyle
 */
public int getStyle() {
	return style;
}

/**
 * Returns an integer hash code for the receiver. Any two 
 * objects which return <code>true</code> when passed to 
 * <code>equals</code> must return the same value for this
 * method.
 *
 * @return the receiver's hash
 *
 * @see #equals
 */
public int hashCode () {
    return
        name.hashCode() ^
        height ^
        style ^
        szFacename.hashCode() ^
        fsSelection;
}

/**
 * Sets the height of the receiver. The parameter is
 * specified in terms of points, where a point is one
 * seventy-second of an inch.
 *
 * @param height the height of the <code>FontData</code>
 *
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_INVALID_ARGUMENT - if the height is negative</li>
 * </ul>
 * 
 * @see #getHeight
 */
public void setHeight(int height) {
	if (height < 0) SWT.error(SWT.ERROR_INVALID_ARGUMENT);
	this.height = height;
}

/**
 * Sets the locale of the receiver.
 * <p>
 * The locale determines which platform character set this
 * font is going to use. Widgets and graphics operations that
 * use this font will convert UNICODE strings to the platform
 * character set of the specified locale.
 * </p>
 * <p>
 * On platforms which there are multiple character sets for a
 * given language/country locale, the variant portion of the
 * locale will determine the character set.
 * </p>
 * 
 * @param locale the <code>String</code> representing a Locale object
 * @see java.util.Locale#toString
 */
public void setLocale(String locale) {
    /*
     *  Currently this function does nothing in OS/2 due to some system
     *  limitations and bugs. This will be changed in the future.
     */	
	lang = country = variant = null;
	if (locale != null) {
		char sep = '_';
		int length = locale.length();
		int firstSep, secondSep;
		
		firstSep = locale.indexOf(sep);
		if (firstSep == -1) {
			firstSep = secondSep = length;
		} else {
			secondSep = locale.indexOf(sep, firstSep + 1);
			if (secondSep == -1) secondSep = length;
		}
		if (firstSep > 0) lang = locale.substring(0, firstSep);
		if (secondSep > firstSep + 1) country = locale.substring(firstSep + 1, secondSep);
		if (length > secondSep + 1) variant = locale.substring(secondSep + 1);
	}
}

/**
 * Sets the name of the receiver.
 * <p>
 * Some platforms support font foundries. On these platforms, the name
 * of the font specified in setName() may have one of the following forms:
 * <ol>
 * <li>a face name (for example, "courier")</li>
 * <li>a foundry followed by a dash ("-") followed by a face name (for example, "adobe-courier")</li>
 * </ol>
 * In either case, the name returned from getName() will include the
 * foundry.
 * </p>
 * <p>
 * On platforms that do not support font foundries, only the face name
 * (for example, "courier") is used in <code>setName()</code> and 
 * <code>getName()</code>.
 * </p>
 *
 * @param name the name of the font data (must not be null)
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_NULL_ARGUMENT - when the font name is null</li>
 * </ul>
 *
 * @see #getName
 */
public void setName(String name) {
	if (name == null) SWT.error(SWT.ERROR_NULL_ARGUMENT);
    this.name = name;
    szFacename = "";
}

/**
 * Sets the style of the receiver to the argument which must
 * be a bitwise OR of one or more of the <code>SWT</code> 
 * constants NORMAL, BOLD and ITALIC.
 *
 * @param style the new style for this <code>FontData</code>
 *
 * @see #getStyle
 */
public void setStyle(int style) {
    this.style = (style & (SWT.BOLD | SWT.ITALIC)); 
    szFacename = "";
}

/**
 * Returns a string representation of the receiver which is suitable
 * for constructing an equivalent instance using the 
 * <code>FontData(String)</code> constructor.
 *
 * @return a string representation of the FontData
 *
 * @see FontData
 */
public String toString() {
	StringBuffer buffer = new StringBuffer();
	buffer.append("1|");
	buffer.append(getName());
	buffer.append("|");
	buffer.append(getHeight());
	buffer.append("|");
	buffer.append(getStyle());
	buffer.append("|");
	buffer.append("OS2|1|");	
	buffer.append(szFacename);
	buffer.append("|");
	buffer.append(fsSelection);
	return buffer.toString();
}

/**	 
 * Invokes platform specific functionality to allocate a new font data.
 * <p>
 * <b>IMPORTANT:</b> This method is <em>not</em> part of the public
 * API for <code>FontData</code>. It is marked public only so that
 * it can be shared within the packages provided by SWT. It is not
 * available on all platforms, and should never be called from
 * application code.
 * </p>
 *
 * @param handle the <code>FATTRS</code> for the font data
 * 
 * @private
 */
public static FontData pm_new(FATTRS handle) {
    FontData fd = new FontData();
    fd.name = handle.fm_szFamilyname;
    fd.height = handle.height;
    fd.style = SWT.NORMAL;
    if (handle.fm_usWeightClass > 5) fd.style |= SWT.BOLD;
    if ((handle.fm_fsSelection & OS.FM_SEL_ITALIC) != 0) fd.style |= SWT.ITALIC;
    fd.szFacename = new PSZ (handle.szFacename).toString();
    fd.fsSelection = handle.fsSelection;
	return fd;
}
/**	 
 * Invokes platform specific functionality to return an OS font stucture.
 * <p>
 * <b>IMPORTANT:</b> This method is <em>not</em> part of the public
 * API for <code>FontData</code>. It is marked public only so that
 * it can be shared within the packages provided by SWT. It is not
 * available on all platforms, and should never be called from
 * application code.
 * </p>
 *
 * @return an instance of an OS/2 font structure as a Java object
 * 
 * @private
 */
public FATTRS getFAttrs() {
		 FATTRS fAttrs = new FATTRS();
		 fAttrs.szFacename = (new PSZ(this.szFacename)).getBytes();
		 fAttrs.height = this.height;
		 fAttrs.fm_usWeightClass = (short)((this.style & SWT.BOLD) != 0 ?7:5);
		 fAttrs.fm_fsSelection = (short)((this.style & SWT.ITALIC) != 0 ? OS.FM_SEL_ITALIC:0);
		 //@@TODO (lpino): remove    
		 // if((fontData.getStyle() & SWT.ITALIC) != 0)
			 // fAttrs.fm_fsSelection |= OS.FM_SEL_ITALIC;
		 // if((fontData.getStyle() & SWT.BOLD) != 0)
			 // fAttrs.fm_fsSelection |= OS.FM_SEL_BOLD;
		 fAttrs.fsSelection = this.fsSelection;
                 fAttrs.usRecordLength = FATTRS.sizeof;
		 return fAttrs;
}
}

